<#
.Description
This is a command script for Actifile cloud scanner.
.PARAMETER Command
Deploy - deploy and run a VM in Azure cloud with Actifile cloud scanner.
Start - start a previously stopped VM. During startup the script will check for updated cloud scanner and install if needed.
Stop - stop and deallocate the VM.
Remove - delete the VM.
Status - get current VM status and print last logs from cloud scanner.
Update-Config - update configuration files for cloud scanner in a running VM.
Remove-Config - remove configuration files for cloud scanner in a running VM.

.PARAMETER SelectContext
Starts a browser authentication popup to connect to an Azure instance. 
.PARAMETER ResourceGroupName
Select a resource group name in Azure cloud
.PARAMETER VmName
VM name in Azure cloud 
.PARAMETER OneDriveConfigFile
Path to configuration file for Microsoft OneDrive and SharePoint, downloaded from https://app.actifile.com/Tenant/Default/#/GeneralSettings
.PARAMETER GoogleDriveConfigFile
Path to configuration file for Google Drive, downloaded from https://app.actifile.com/Tenant/Default/#/GeneralSettings
.PARAMETER OutlookConfigFile
Path to configuration file for Microsoft Outlook, downloaded from https://app.actifile.com/Tenant/Default/#/GeneralSettings
.PARAMETER GoogleMailConfigFile
Path to configuration file for Google GMail, downloaded from https://app.actifile.com/Tenant/Default/#/GeneralSettings
.PARAMETER UserName
User name in VM
.PARAMETER PublicIpAddressName
Public Ip address name in Azure cloud
.PARAMETER SshKeyName
SSH key name 
.PARAMETER DoNotRemoveSshKey
Do not remove SSH key when removing VM
.PARAMETER Force
Forces the command to run without asking for user confirmation.
.EXAMPLE
# Deploy the Actifile cloud scanner using a OneDrive configuration file and forve to start a browser authentication popup to connect to an Azure instance.
PS> .\actifile-cloud-scanner -OneDriveConfigFile .\actifile_onedrive.conf -Command Deploy -SelectContext
.EXAMPLE
# Deploy the Actifile cloud scanner using a Google Drive configuration file.
PS> .\actifile-cloud-scanner -GoogleDriveConfigFile .\actifile_googledrive.conf -Command Deploy -SelectContext
.EXAMPLE
# Deploy the Actifile cloud scanner using both OneDrive and Google Drive configurations.
PS> .\actifile-cloud-scanner -OneDriveConfigFile .\actifile_onedrive.conf -GoogleDriveConfigFile .\actifile_googledrive.conf -Command Deploy -SelectContext
.EXAMPLE
# Check the status of the deployed scanner.
PS> .\actifile-cloud-scanner -Command Status
.EXAMPLE
# Stop the Actifile cloud scanner VM.
PS> .\actifile-cloud-scanner -Command Stop
.EXAMPLE
# Remove the Actifile cloud scanner VM.
PS> .\actifile-cloud-scanner -Command Remove -DoNotRemoveSshKey -Force
.EXAMPLE
# Upadte existing or add new configuration files to a running VM.
PS> .\actifile-cloud-scanner -Command Update-Config -OneDriveConfigFile .\actifile_onedrive.conf
.EXAMPLE
# Remove scanner configuration from a running VM.
PS> .\actifile-cloud-scanner -Command Remove-Config -Type OneDrive
.EXAMPLE
# List all configuration files in the VM.
PS> .\actifile-cloud-scanner -Command List-Configurations
.SYNOPSIS
This Powershell script is used to deploy Actifile's cloud scanning agent on a virtual machine on Azure. 
.LINK
https://apptest.actifile.com/Tenant/Default/#/GeneralSettings

#> 

[CmdletBinding(DefaultParameterSetName = 'Command')]
param (
   [string]$VmName = "ActifileCloudScanner",
   [string]$UserName = "azureuser",
   [string]$OneDriveConfigFile,
   [string]$OutlookConfigFile,
   [string]$GoogleDriveConfigFile,
   [string]$GoogleMailConfigFile,
   [parameter(ParameterSetName = 'Command')][ValidateSet("Start", "Status", "Stop", "Remove", "Deploy", "Update-Config", "Remove-Config", "List-Configurations")][string]$Command,
   [string]$PublicIpAddressName,
   [string]$SshKeyName,
   [string]$ResourceGroupName,
   [switch]$SelectContext = $false,
   [switch]$DoNotRemoveSshKey = $false,
   [switch]$Force = $false,
   [switch]$UseExistingVm = $false
)
    
DynamicParam {
   $paramDictionary = New-Object -TypeName System.Management.Automation.RuntimeDefinedParameterDictionary
        
   if ($PSCmdlet.ParameterSetName -eq 'Command' -and $Command -eq 'Remove-Config') {
      $attributes = New-Object System.Management.Automation.ParameterAttribute
      $attributes.Mandatory = $true
      $attributes.Position = 0
      $attributes.HelpMessage = "Specify the configuration to remove: OneDrive, Outlook, GoogleDrive, GoogleMail"
      $validateSet = New-Object System.Management.Automation.ValidateSetAttribute("OneDrive", "Outlook", "GoogleDrive", "GoogleMail")
      $attributeCollection = New-Object System.Collections.ObjectModel.Collection[System.Attribute]
      $attributeCollection.Add($attributes)
      $attributeCollection.Add($validateSet)
      $param = New-Object System.Management.Automation.RuntimeDefinedParameter('Type', [string], $attributeCollection)
      $paramDictionary.Add('Type', $param)
   }
        
   return $paramDictionary
}

begin {
   $OneDriveConfigFileDefault = "actifile_onedrive.conf"
   $OutlookConfigFileDefault = "actifile_outlook.conf"
   $GoogleDriveConfigFileDefault = "actifile_googledrive.conf"
   $GoogleMailConfigFileDefault = "actifile_googlemail.conf"
    
   if (-not $PublicIpAddressName) { $PublicIpAddressName = $VmName + "PubIP" }
   if (-not $SshKeyName) { $SshKeyName = $VmName + "SshKeys" }
   $currentVersion = "0.9.9.20"
   $scriptName = $MyInvocation.MyCommand.Name
   $versionURL = "https://app.actifile.com/AgentNew/Release/Win/Install/deploy_agent_script_version.txt"
   $latestVersionURL = "https://app.actifile.com/AgentNew/Release/Win/Install/$scriptName.zip"
   $DockerRegistry = "docker.io"
   $DockerImageName = "actifile/cloudscanner:latest"
   $DockerContainerName = "cloudscanner_agent"
   $DockerCmd = "docker run -d --name $DockerContainerName --restart unless-stopped --log-driver=journald --log-opt=mode=non-blocking -v actifile:/conf --init --hostname=$VmName $DockerRegistry/$DockerImageName"
   $DockerWatchTowerCmd = "docker run -d --name watchtower --restart unless-stopped -v /var/run/docker.sock:/var/run/docker.sock $DockerRegistry/actifile/watchtower -i 1800 --cleanup"
   $TempFolderPath = $env:TEMP
   $requiredModules = @{
      "Az.Compute"   = "7.3.0"
      "Az.Network"   = "7.5.0"
      "Az.Resources" = "6.6.0"
      "Az.Accounts"  = "2.19.0"
   }

   Set-Item Env:\SuppressAzurePowerShellBreakingChangeWarnings "true"
   $ErrorActionPreference = "Stop"
}
process {
   Write-Verbose "VM Name: $VmName"
   Write-Verbose "User Name: $UserName"
   Write-Verbose "OneDrive Config File: $OneDriveConfigFile"
   Write-Verbose "Outlook Config File: $OutlookConfigFile"
   Write-Verbose "Google Drive Config File: $GoogleDriveConfigFile"
   Write-Verbose "Google Mail Config File: $GoogleMailConfigFile"
   Write-Verbose "Command: $Command"
   Write-Verbose "Public IP Address Name: $PublicIpAddressName"
   Write-Verbose "SSH Key Name: $SshKeyName"
   Write-Verbose "Resource Group Name: $ResourceGroupName"
   Write-Verbose "Select Context: $SelectContext"
   Write-Verbose "Do Not Remove SSH Key: $DoNotRemoveSshKey"
   Write-Verbose "Force: $Force"
   Write-Verbose "Use Existing VM: $UseExistingVm"
   Write-Verbose "currrent version: $currentVersion"

   $AllParameters = "";
   ForEach ($Parameter in $PsBoundParameters.GetEnumerator()) {
      $Parameter_Key = $Parameter.Key;
      $Parameter_Value = $Parameter.Value;
      $Parameter_Value_Type = $Parameter_Value.GetType().Name;

      If ($Parameter_Value_Type -Eq "SwitchParameter") {
         $AllParameters += " -$Parameter_Key";
      }
      Else {
         $AllParameters += " -$Parameter_Key $Parameter_Value";
      }
   }
   function CheckForUpdates ($AllParameters) {
      try {
         $latestVersion = Invoke-WebRequest -Uri $versionURL -UseBasicParsing | Select-Object -ExpandProperty Content
         $latestVersion = $latestVersion.Trim()
        
         if ([version]$currentVersion -lt [version]$latestVersion) {
            Write-Host "An update is available! Latest version: $latestVersion"
            Write-Host "Please download the latest version from: $latestVersionURL"
            $response = Read-Host "Would you like to download and install the latest version? (Y/N)"
            if ($response -eq "Y" -or $response -eq "y") {
               if (-not $TempFolderPath -and $IsLinux ) {
                  $TempFolderPath = "/tmp"
               }
               $zipFilePath = "$TempFolderPath\latest_version.zip"
               Invoke-WebRequest -Uri $latestVersionURL -OutFile $zipFilePath

               Expand-Archive -Path $zipFilePath -DestinationPath $TempFolderPath\ActifileUpdatedScript -Force

               Copy-Item -Path "$TempFolderPath\ActifileUpdatedScript\$scriptName" -Destination "$PSCommandPath" -Force

               Remove-Item -Path $zipFilePath -Force
               Remove-Item -Path "$TempFolderPath\ActifileUpdatedScript" -Recurse -Force

               Write-Host "Update successful. Restarting the script..."
               Start-Sleep -Seconds 2

               Write-Host "Restarting the script..."
               & $PSCommandPath $AllParameters
               Exit
            }
         }
         else {
            Write-Verbose "You have the latest version."         
         }
      }
      catch {
         Write-Verbose "Failed to check for updates. Error: $_"
      }
   }
   function PromtYN {
      param (
         [string]$Message
      )
      if (-not $Force) {
         do {
            $answer = Read-Host -Prompt "$Message (y/n)"
            $ok = $FALSE
            switch ($answer) {
               'y' { $ok = $TRUE; break }
               'n' { exit 1 }
               default { continue }
            }
         }
         while (-not $ok)
      }
   }
   function Copy-ConfigFile {
      param (
         [string]$ResourceGroupName,
         [string]$VmName,
         [string]$ConfigFileSrc,
         [string]$ConfigFileDst
      )
      if (-not (Test-Path -Path $ConfigFileSrc)) {
         throw "Can't find configuration file $ConfigFileSrc"
      }
      $content = (Get-Content $ConfigFileSrc)
      Write-Host "Transfering $ConfigFileSrc to VM"
      Write-Host "Please wait..."
      $cmd = "sudo echo '$content' > `$(docker volume inspect --format '{{ .Mountpoint }}' actifile)/$ConfigFileDst"         
      $output = Invoke-AzVMRunCommand `
         -ResourceGroupName $ResourceGroupName `
         -Name $VmName `
         -CommandId 'RunShellScript' `
         -ScriptString "$cmd" -Verbose
      Write-Host $output.Value.Message
   }
   function ShowConfigFiles {
      param (
         [string]$ResourceGroupName,
         [string]$VmName
      )
      Write-Verbose "List configuration files in VM $VmName"
      Write-Host "Please wait..."
      $cmd = "sudo ls `$(docker volume inspect --format '{{ .Mountpoint }}' actifile)/*.conf"
      Write-Verbose "Executing command: $cmd"
      $output = Invoke-AzVMRunCommand `
         -ResourceGroupName $ResourceGroupName `
         -Name $VmName `
         -CommandId 'RunShellScript' `
         -ScriptString "$cmd"
      Write-Host $output.Value.Message 
   }
   function CheckVMIsRunning {
      param (
         [string]$ResourceGroupName,
         [string]$VmName
      )
      $VmStatus = Get-AzVm -Name $VmName -ResourceGroupName $ResourceGroupName -Status -ErrorAction SilentlyContinue
      if ($VmStatus) {
         if ($VmStatus.Statuses.DisplayStatus[1] -eq 'VM running') {
            return $TRUE
         }
         else {
            Write-Host "VM exists but is not started. Please, start it first."
         }
      }
      else {
         Write-Host "VM does not exist."
      }    
   }


   function Remove-ConfigFile {
      param (
         [string]$ResourceGroupName,
         [string]$VmName,
         [string]$ConfigFile
      )
      Write-Host "Removing $ConfigFile from VM $VmName"
      Write-Host "Please wait..."
      $cmd = "sudo rm `$(docker volume inspect --format '{{ .Mountpoint }}' actifile)/$ConfigFile"
      Write-Verbose "Executing command: $cmd"
      $output = Invoke-AzVMRunCommand `
         -ResourceGroupName $ResourceGroupName `
         -Name $VmName `
         -CommandId 'RunShellScript' `
         -ScriptString "$cmd" -Verbose
      Write-Host $output.Value.Message 
   }

   function Restart-Container {
      param (
         [string]$ResourceGroupName,
         [string]$VmName
      )
      Write-Host "Restarting container, Please wait..."
      $cmd = "docker restart $DockerContainerName"
      Write-Verbose "Executing command: $cmd"
      $output = Invoke-AzVMRunCommand `
         -ResourceGroupName $ResourceGroupName `
         -Name $VmName `
         -CommandId 'RunShellScript' `
         -ScriptString "$cmd"
      Write-Host $output.Value.Message
   }
   function Stop-Container {
      param (
         [string]$ResourceGroupName,
         [string]$VmName
      )
      Write-Host "Stopping container, Please wait..."
      $cmd = "docker stop $DockerContainerName"
      Write-Verbose "Executing command: $cmd"
      $output = Invoke-AzVMRunCommand `
         -ResourceGroupName $ResourceGroupName `
         -Name $VmName `
         -CommandId 'RunShellScript' `
         -ScriptString "$cmd"
      Write-Host $output.Value.Message
   }
   function Start-Container {
      param (
         [string]$ResourceGroupName,
         [string]$VmName
      )
      Write-Host "Starting container, Please wait..."
      $cmd = "docker start $DockerContainerName"
      Write-Verbose "Executing command: $cmd"
      $output = Invoke-AzVMRunCommand `
         -ResourceGroupName $ResourceGroupName `
         -Name $VmName `
         -CommandId 'RunShellScript' `
         -ScriptString "$cmd"
      Write-Host $output.Value.Message
   }
   function Show-Menu {
      param (
         [string]$Title = 'Choose',
         [array]$Options
      )
      if ($Options.Count -eq 1) {
         return 0
      }
      do {
         Write-Host "================ $Title ================"
         for ($i = 0; $i -lt $Options.length; $i++ ) {
            Write-Host "Press '"($i + 1)"' for " $Options[$i]
         }
         $res = Read-Host
      }while ($res -gt $Options.Length -or $res -le 0)
      return $res - 1
   }
   Write-Host "PowerShell version" $PSVersionTable.PSVersion.ToString()

   CheckForUpdates($AllParameters)
   Write-Verbose "$scriptName version $currentV ersion"

   $missingModules = @()

   foreach ($module in $requiredModules.Keys) {
      $installedModule = Get-Module -Name $module -ListAvailable | Sort-Object Version -Descending | Select-Object -First 1
      if ($installedModule -and ($installedModule.Version -ge [version]$requiredModules[$module])) {
         continue
      }
      elseif ($installedModule) {
         Write-Host "$module is installed but does not meet the minimum version requirement. Required: $($requiredModules[$module]), Installed: $($installedModule.Version)"
      }
      else {
         Write-Host "$module is not installed."
      }
      $missingModules += $module
   }
   if ($missingModules.Count -gt 0) {
      Write-Host "One or more required modules are not installed. Please install the required modules and try again."
      Write-Host "For detailed instructions follow the link:"
      Write-Host "https://learn.microsoft.com/en-us/powershell/azure/install-azure-powershell?view=azps-12.2.0"
      Write-Host "Example command to install missing modules:"
      foreach ($mod in $missingModules) {
         # -RequiredVersion $($modules[$mod]) 
         Write-Host "Install-Module -Name $mod -Scope CurrentUser -Repository PSGallery -Force -AllowClobber"
      }
      return
   }

   if ($Command -ne "") {
      if ($SelectContext -or ($null -eq (Get-AzContext).Tenant.Id )) {
         Connect-AzAccount
      }
      else {
         Write-Host "Using Azure context: " + (Get-AzContext).Name
      }
      $AccessToken = Get-AzAccessToken -ErrorAction SilentlyContinue
      if (!$AccessToken) {
         Write-Host "Login needed"
         try {
            Login-AzAccount -ErrorAction stop
         }
         catch {
            throw "Could not login to Azure"
         }
      }
      if ($ResourceGroupName -eq "") {
         if (@(Get-AzResourceGroup -ErrorAction SilentlyContinue).Count -eq 0) {
            Write-Warning "Can't find any resource group."
            Write-Host "Please, create resource group. For detailed instructions follow the link:  https://portal.azure.com/#create/Microsoft.ResourceGroup"
            exit 1
         }
         else {
            $ResourceGroups = @((Get-AzResourceGroup).ResourceGroupName)
            $r = Show-Menu -Options $ResourceGroups -Title "Choose resource group"
            $ResourceGroupName = $ResourceGroups[$r]
         }
      }

      Write-Host "Resource group: " + $ResourceGroupName
      $Location = (Get-AzResourceGroup -Name $ResourceGroupName).Location

   }

   switch ($Command) {
      "Start" {
         Start-AzVM -ResourceGroupNam $ResourceGroupName -Name $VmName -Verbose

         Write-Host "Pulling the latest version and starting docker container"

         $cmd = @"
      docker pull $DockerRegistry/$DockerImageName
      docker container ls --format '{{.Names}}' | xargs -I {}   docker container rm -f {}
      $DockerCmd
      docker image prune -a -f
"@
         $output = Invoke-AzVMRunCommand `
            -ResourceGroupName $ResourceGroupName `
            -Name $VmName `
            -CommandId 'RunShellScript' `
            -ScriptString $cmd -Verbose 
         Write-Host $output.Value.Message

         $output = Invoke-AzVMRunCommand `
            -ResourceGroupName $ResourceGroupName `
            -Name $VmName `
            -CommandId 'RunShellScript' `
            -ScriptString $DockerWatchTowerCmd -Verbose
         Write-Host $output.Value.Message

         $output = Invoke-AzVMRunCommand `
            -ResourceGroupName $ResourceGroupName `
            -Name $VmName `
            -CommandId 'RunShellScript' `
            -ScriptString 'docker container ls -a'
         Write-Host $output.Value.Message

         Write-Host Started with IP: 
         $VM = Get-AzVm -name $VmName -ResourceGroupName $ResourceGroupName
         $NetworkPofile = $VM.NetworkProfile.NetworkInterfaces.Id.Split("/") | Select-Object -Last 1
         $IPConfig = Get-AzNetworkInterface -Name $NetworkPofile
         $IPAddressName = $IPConfig.IpConfigurations.PublicIpAddress.Id.Split("/") | Select-Object -Last 1
      (Get-AzPublicIpAddress -Name $IPAddressName).IpAddress
         break
      }
   
      "Stop" {
         if (-not $Force) {
            do {
               $answer = Read-Host -Prompt "Stop VM $VmName (y/n)"
               $ok = $FALSE
               switch ($answer) {
                  'y' { $ok = $TRUE; break }
                  'n' { exit 1 }
                  default { continue }
               }
            }
            while (-not $ok)
         }

         Write-Host "Please wait..."
         Stop-AzVM -ResourceGroupNam $ResourceGroupName -Name $VmName -Force
         break
      }

      "Deploy" {
         if ((-not (Test-Path -Path $OneDriveConfigFile)) -and (-not (Test-Path -Path $GoogleDriveConfigFile)) -and (-not (Test-Path -Path $OutlookConfigFile)) -and (-not (Test-Path -Path $GoogleMailConfigFile))) {
            throw "Can't find any configuration file"
         }
         if (Test-Path -Path $OneDriveConfigFile) {
            Write-Host "Using config file for OneDrive/SharePoint: $OneDriveConfigFile"
            $useOneDrive = $TRUE
         }
         if (Test-Path -Path $OutlookConfigFile) {
            Write-Host "Using config file for Outlook: $OutlookConfigFile"
            $useOutlook = $TRUE
         }
         if (Test-Path -Path $GoogleDriveConfigFile) {
            Write-Host "Using config file for Google Drive: $GoogleDriveConfigFile"
            $useGoogleDrive = $TRUE
         }
         if (Test-Path -Path $GoogleMailConfigFile) {
            Write-Host "Using config file for GMail: $GoogleMailConfigFile"
            $useGoogleMail = $TRUE
         }
         PromtYN -Message 'Continue to deploy?'
      
         if (-not $UseExistingVm) {
            $Password = Read-Host -AsSecureString -Prompt "Password for user 'azureuser' in deployed virtual machine(could be empty)"
            $Credential = New-Object -TypeName System.Management.Automation.PSCredential -ArgumentList $UserName, $Password

            Write-Host "Creating new vitrual machine"
            Write-Host "PublicIpAddressName: $PublicIpAddressName"
            Write-Host "SshKeyName: $SshKeyName"
            $Image = "Ubuntu2204"
            $Size = "Standard_DS1_v2"
            if (@(Get-AzSshKey -Name $SshKeyName -ResourceGroupName $ResourceGroupName  -ErrorAction SilentlyContinue).Count -eq 0) {
               New-AzVm `
                  -PublicIpSku Standard `
                  -ResourceGroupName $ResourceGroupName `
                  -Credential $Credential `
                  -Name $VmName `
                  -OSDiskDeleteOption Delete `
                  -DataDiskDeleteOption Delete `
                  -NetworkInterfaceDeleteOption Delete `
                  -Location $Location `
                  -PublicIpAddressName $PublicIpAddressName `
                  -Image $Image `
                  -Size $Size `
                  -GenerateSshKey `
                  -SshKeyName $SshKeyName
            }
            else {
               Write-Host "Using existed SSH keys."
               New-AzVm `
                  -PublicIpSku Standard `
                  -ResourceGroupName $ResourceGroupName `
                  -Credential $Credential `
                  -Name $VmName `
                  -OSDiskDeleteOption Delete `
                  -NetworkInterfaceDeleteOption Delete `
                  -DataDiskDeleteOption Delete `
                  -Location $Location `
                  -PublicIpAddressName $PublicIpAddressName `
                  -Image $Image `
                  -Size $Size `
                  -SshKeyName $SshKeyName
            }
         }

         $cmd = @"
export DEBIAN_FRONTEND=noninteractive 
sudo apt-get update 
sudo apt-get install -y  docker.io && sudo usermod -aG docker azureuser && docker volume create actifile && echo OK
"@
         $output = Invoke-AzVMRunCommand `
            -ResourceGroupName $ResourceGroupName `
            -Name $VmName `
            -CommandId 'RunShellScript' `
            -ScriptString "$cmd"
         Write-Host $output.Value.Message
         if ($useOneDrive) {
            Copy-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFileSrc $OneDriveConfigFile -ConfigFileDst $OneDriveConfigFileDefault
         }
         if ($useOutlook) {
            Copy-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFileSrc $OutlookConfigFile -ConfigFileDst $OutlookConfigFileDefault
         }
         if ($useGoogleDrive) {
            Copy-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFileSrc $GoogleDriveConfigFile -ConfigFileDst $GoogleDriveConfigFileDefault
         }
         if ($useGoogleMail) {
            Copy-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFileSrc $GoogleMailConfigFile -ConfigFileDst $GoogleMailConfigFileDefault
         }

         Write-Host "Starting Actifile cloud scanner container"
         $output = Invoke-AzVMRunCommand `
            -ResourceGroupName $ResourceGroupName `
            -Name $VmName `
            -CommandId 'RunShellScript' `
            -ScriptString "$DockerCmd"
         Write-Host $output.Value.Message

         Write-Host "Starting watchtower container"
         $output = Invoke-AzVMRunCommand `
            -ResourceGroupName $ResourceGroupName `
            -Name $VmName `
            -CommandId 'RunShellScript' `
            -ScriptString "$DockerWatchTowerCmd"
         Write-Host $output.Value.Message

         Write-Host "Started VM with IP: " 

         $VM = Get-AzVm -name $VmName
         $NetworkPofile = $VM.NetworkProfile.NetworkInterfaces.Id.Split("/") | Select-Object -Last 1
         $IPConfig = Get-AzNetworkInterface -Name $NetworkPofile
         $IPAddressName = $IPConfig.IpConfigurations.PublicIpAddress.Id.Split("/") | Select-Object -Last 1
         Write-Host (Get-AzPublicIpAddress -Name $IPAddressName).IpAddress
         break
      }

      "Remove" {
         Write-Host "Removing virtual machine $VmName"
         if ($Force) {
            Remove-AzVM -Name $VmName -ResourceGroupName $ResourceGroupName -Force 
         }
         else {
            Remove-AzVM -Name $VmName -ResourceGroupName $ResourceGroupName 
         }
         if (-not $DoNotRemoveSshKey) {
            Write-Host "Removing ssh keys for $VmName"
            Remove-AzSshKey -Name $SshKeyName -ResourceGroupName $ResourceGroupName
         }
      }

      "Status" {
         $PowerState = (Get-AzVM -VMName $VmName -Status).PowerState
         if ( $null -eq $PowerState) {
            Write-Host "VM '$VmName' doesn't exist"
            exit 0
         }
         if ($PowerState -ne "VM running") {
            Write-Host "PowerState is " + $PowerState
            exit 0
         }
         Write-Host "Please wait..."
         $output = Invoke-AzVMRunCommand `
            -ResourceGroupName $ResourceGroupName `
            -Name $VmName `
            -CommandId 'RunShellScript' `
            -ScriptString 'docker container ls -a && docker image ls'
         Write-Host $output.Value.Message
         $script = @'
               echo "existed containers $(docker container ls -a)"
               container_id=$DockerContainerName
               cmd="/app/agent.py"
               if docker container exec $container_id [ -f "/conf/actifile_onedrive.conf" ]; then
                     cmd="$cmd --onedrive-config /conf/actifile_onedrive.conf"
               fi
               if docker container exec $container_id [ -f "/conf/actifile_outlook.conf" ]; then
                     cmd="$cmd --outlook-config /conf/actifile_outlook.conf"
               fi
               if docker container exec $container_id [ -f "/conf/actifile_googledrive.conf" ]; then
                     cmd="$cmd --google-drive-config /conf/actifile_googledrive.conf"
               fi
               if docker container exec $container_id [ -f "/conf/actifile_googlemail.conf" ]; then
                     cmd="$cmd --gmail-config /conf/actifile_googlemail.conf"
               fi
               docker container exec $container_id $cmd --info
  
'@
         $script = $script -replace '\$DockerContainerName', $DockerContainerName
         $output = Invoke-AzVMRunCommand `
            -ResourceGroupName $ResourceGroupName `
            -Name $VmName `
            -CommandId 'RunShellScript' `
            -ScriptString $script
         Write-Host $output.Value.Message
         Write-Host "docker logs $DockerContainerName --tail 1000"
         $output = Invoke-AzVMRunCommand `
            -ResourceGroupName $ResourceGroupName `
            -Name $VmName `
            -CommandId 'RunShellScript' `
            -ScriptString "docker logs $DockerContainerName --tail 1000"
         Write-Host $output.Value.Message

         Write-Host "Host IP:"
         $VM = Get-AzVm -name $VmName
         $NetworkPofile = $VM.NetworkProfile.NetworkInterfaces.Id.Split("/") | Select-Object -Last 1
         $IPConfig = Get-AzNetworkInterface -Name $NetworkPofile
         $IPAddressName = $IPConfig.IpConfigurations.PublicIpAddress.Id.Split("/") | Select-Object -Last 1

         (Get-AzPublicIpAddress -Name $IPAddressName).IpAddress
         break
      }  
      "Update-Config" {
         if (CheckVMIsRunning -ResourceGroupName $ResourceGroupName -VmName $VmName) {
            Write-Host "VM exists and is started. Updating configuration files."
            if (Test-Path -Path $OneDriveConfigFile) {
               Write-Host "Using config file for OneDrive/SharePoint: $OneDriveConfigFile"
               $useOneDrive = $TRUE
            }
            if (Test-Path -Path $OutlookConfigFile) {
               Write-Host "Using config file for Outlook: $OutlookConfigFile"
               $useOutlook = $TRUE
            }
            if (Test-Path -Path $GoogleDriveConfigFile) {
               Write-Host "Using config file for Google Drive: $GoogleDriveConfigFile"
               $useGoogleDrive = $TRUE
            }
            if (Test-Path -Path $GoogleMailConfigFile) {
               Write-Host "Using config file for GMail: $GoogleMailConfigFile"
               $useGoogleMail = $TRUE
            }
            if (-not $useOneDrive -and -not $useOutlook -and -not $useGoogleDrive -and -not $useGoogleMail) {
               throw "Can't find any configuration file"
            }
            PromtYN -Message 'Continue to update configuration?'
            Stop-Container -ResourceGroupName $ResourceGroupName -VmName $VmName
            if ($useOneDrive) {
               Copy-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFileSrc $OneDriveConfigFile -ConfigFileDst $OneDriveConfigFileDefault
            }
            if ($useOutlook) {
               Copy-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFileSrc $OutlookConfigFile -ConfigFileDst $OutlookConfigFileDefault
            }
            if ($useGoogleDrive) {
               Copy-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFileSrc $GoogleDriveConfigFile -ConfigFileDst $GoogleDriveConfigFileDefault
            }
            if ($useGoogleMail) {
               Copy-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFileSrc $GoogleMailConfigFile -ConfigFileDst $GoogleMailConfigFileDefault
            }
            Start-Container -ResourceGroupName $ResourceGroupName -VmName $VmName 
         }
         break
      }
      "Remove-Config" {
         $ScannerType = $PSBoundParameters['Type']
         Write-Host "Removing scanner configuration : $ScannerType"
         if (CheckVMIsRunning -ResourceGroupName $ResourceGroupName -VmName $VmName) {
            PromtYN -Message "Remove configuration $ScannerType from $VmName`?"
            switch ($ScannerType) {
               "OneDrive" {
                  Remove-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFile $OneDriveConfigFileDefault
               }
               "Outlook" {
                  Remove-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFile $OutlookConfigFileDefault
               }
               "GoogleDrive" {
                  Remove-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFile $GoogleDriveConfigFileDefault
               }
               "GoogleMail" {
                  Remove-ConfigFile -ResourceGroupName $ResourceGroupName -VmName $VmName -ConfigFile $GoogleMailConfigFileDefault
               }
            }
            Restart-Container -ResourceGroupName $ResourceGroupName -VmName $VmName 
         }
         break
      }
      "List-Configurations" {
         if (CheckVMIsRunning -ResourceGroupName $ResourceGroupName -VmName $VmName) {
            ShowConfigFiles -ResourceGroupName $ResourceGroupName -VmName $VmName
         }
         break
      }
      default {
         Write-Host "Unrecognized option."
         Write-Host "Try Get-Help .\actifile-cloud-scanner.ps1 -Full for more information."
      }
   }
}

# SIG # Begin signature block
# MIIgzgYJKoZIhvcNAQcCoIIgvzCCILsCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCATUD5tSailvrOv
# aeigCqCOzvqljJDMhRJb/tfW0h2WLKCCB4IwggNZMIIC36ADAgECAhAPuKdAuRWN
# A1FDvFnZ8EApMAoGCCqGSM49BAMDMGExCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxE
# aWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xIDAeBgNVBAMT
# F0RpZ2lDZXJ0IEdsb2JhbCBSb290IEczMB4XDTIxMDQyOTAwMDAwMFoXDTM2MDQy
# ODIzNTk1OVowZDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MTwwOgYDVQQDEzNEaWdpQ2VydCBHbG9iYWwgRzMgQ29kZSBTaWduaW5nIEVDQyBT
# SEEzODQgMjAyMSBDQTEwdjAQBgcqhkjOPQIBBgUrgQQAIgNiAAS7tKwnpUgNolNf
# jy6BPi9TdrgIlKKaqoqLmLWx8PwqFbu5s6UiL/1qwL3iVWhga5c0wWZTcSP8GtXK
# IA8CQKKjSlpGo5FTK5XyA+mrptOHdi/nZJ+eNVH8w2M1eHbk+HejggFXMIIBUzAS
# BgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBSbX7A2up0GrhknvcCgIsCLizh3
# 7TAfBgNVHSMEGDAWgBSz20ik+aHF2K42QcwRY2liKbxLxjAOBgNVHQ8BAf8EBAMC
# AYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMwdgYIKwYBBQUHAQEEajBoMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQAYIKwYBBQUHMAKGNGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEdsb2JhbFJvb3RHMy5jcnQw
# QgYDVR0fBDswOTA3oDWgM4YxaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lD
# ZXJ0R2xvYmFsUm9vdEczLmNybDAcBgNVHSAEFTATMAcGBWeBDAEDMAgGBmeBDAEE
# ATAKBggqhkjOPQQDAwNoADBlAjB4vUmVZXEB0EZXaGUOaKncNgjB7v3UjttAZT8N
# /5Ovwq5jhqN+y7SRWnjsBwNnB3wCMQDnnx/xB1usNMY4vLWlUM7m6jh+PnmQ5KRb
# qwIN6Af8VqZait2zULLd8vpmdJ7QFmMwggQhMIIDpqADAgECAhAJT6Ahs3DMGSWt
# q/LO0FyIMAoGCCqGSM49BAMDMGQxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdp
# Q2VydCwgSW5jLjE8MDoGA1UEAxMzRGlnaUNlcnQgR2xvYmFsIEczIENvZGUgU2ln
# bmluZyBFQ0MgU0hBMzg0IDIwMjEgQ0ExMB4XDTI0MDcxNjAwMDAwMFoXDTI1MTAx
# MTIzNTk1OVowgakxEzARBgsrBgEEAYI3PAIBAxMCSUwxHTAbBgNVBA8MFFByaXZh
# dGUgT3JnYW5pemF0aW9uMRIwEAYDVQQFEwk1MTUyMTQ0NzYxCzAJBgNVBAYTAklM
# MREwDwYDVQQHEwhIZXJ6bGl5YTEVMBMGA1UEChMMQUNUSUZJTEUgTFREMREwDwYD
# VQQLEwhBY3RpZmlsZTEVMBMGA1UEAxMMQUNUSUZJTEUgTFREMFkwEwYHKoZIzj0C
# AQYIKoZIzj0DAQcDQgAEQoEWtqJFgZsaekogExxeDSrduWTXqt9cd2VqVfW9omvN
# QT2Ti8cSPebltanjENgnbeo5YPn9BrrRZGjwZdblk6OCAfIwggHuMB8GA1UdIwQY
# MBaAFJtfsDa6nQauGSe9wKAiwIuLOHftMB0GA1UdDgQWBBS5ZvVi0Za7VMjfjE9Q
# zP+aeFpaFTA9BgNVHSAENjA0MDIGBWeBDAEDMCkwJwYIKwYBBQUHAgEWG2h0dHA6
# Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAww
# CgYIKwYBBQUHAwMwgasGA1UdHwSBozCBoDBOoEygSoZIaHR0cDovL2NybDMuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0R2xvYmFsRzNDb2RlU2lnbmluZ0VDQ1NIQTM4NDIw
# MjFDQTEuY3JsME6gTKBKhkhodHRwOi8vY3JsNC5kaWdpY2VydC5jb20vRGlnaUNl
# cnRHbG9iYWxHM0NvZGVTaWduaW5nRUNDU0hBMzg0MjAyMUNBMS5jcmwwgY4GCCsG
# AQUFBwEBBIGBMH8wJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNv
# bTBXBggrBgEFBQcwAoZLaHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lD
# ZXJ0R2xvYmFsRzNDb2RlU2lnbmluZ0VDQ1NIQTM4NDIwMjFDQTEuY3J0MAkGA1Ud
# EwQCMAAwCgYIKoZIzj0EAwMDaQAwZgIxALGTxoUqV2UFKd2VEqZ+jnJweaFQ4KT5
# Hws3CBuUpPa8D7Le2pSQ1riFg7FPX8q6PAIxAKJNOOK6e9EAUa19bkgJM6U6eStL
# 17lVlXBnJw3shKv0YXSVeBmMSi/9Rh7Vc3QFXTGCGKIwghieAgEBMHgwZDELMAkG
# A1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMTwwOgYDVQQDEzNEaWdp
# Q2VydCBHbG9iYWwgRzMgQ29kZSBTaWduaW5nIEVDQyBTSEEzODQgMjAyMSBDQTEC
# EAlPoCGzcMwZJa2r8s7QXIgwDQYJYIZIAWUDBAIBBQCgfDAQBgorBgEEAYI3AgEM
# MQIwADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgELMQ4w
# DAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgYcVPeAIyiPoN2UL8giZXmFGq
# ddchSqNOuObahdNGTMcwCwYHKoZIzj0CAQUABEcwRQIgR+TDTr2rNzQONvoPKfhw
# +jMUvkQoUHE5PrqiE3aSX9oCIQD9kO5jLhz0RKa8dLzvA3po37ixkjFPl1NlTNJd
# mnuDEaGCFzowghc2BgorBgEEAYI3AwMBMYIXJjCCFyIGCSqGSIb3DQEHAqCCFxMw
# ghcPAgEDMQ8wDQYJYIZIAWUDBAIBBQAweAYLKoZIhvcNAQkQAQSgaQRnMGUCAQEG
# CWCGSAGG/WwHATAxMA0GCWCGSAFlAwQCAQUABCAOJ6zU3HS0947gC+C2axrsbj5Y
# d+JqvBIztJTfSecUSwIRAJYf0kFPSuyAjq3F5N9lVj4YDzIwMjQxMDMxMDk0NTA4
# WqCCEwMwgga8MIIEpKADAgECAhALrma8Wrp/lYfG+ekE4zMEMA0GCSqGSIb3DQEB
# CwUAMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkG
# A1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3Rh
# bXBpbmcgQ0EwHhcNMjQwOTI2MDAwMDAwWhcNMzUxMTI1MjM1OTU5WjBCMQswCQYD
# VQQGEwJVUzERMA8GA1UEChMIRGlnaUNlcnQxIDAeBgNVBAMTF0RpZ2lDZXJ0IFRp
# bWVzdGFtcCAyMDI0MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAvmpz
# n/aVIauWMLpbbeZZo7Xo/ZEfGMSIO2qZ46XB/QowIEMSvgjEdEZ3v4vrrTHleW1J
# WGErrjOL0J4L0HqVR1czSzvUQ5xF7z4IQmn7dHY7yijvoQ7ujm0u6yXF2v1CrzZo
# pykD07/9fpAT4BxpT9vJoJqAsP8YuhRvflJ9YeHjes4fduksTHulntq9WelRWY++
# TFPxzZrbILRYynyEy7rS1lHQKFpXvo2GePfsMRhNf1F41nyEg5h7iOXv+vjX0K8R
# hUisfqw3TTLHj1uhS66YX2LZPxS4oaf33rp9HlfqSBePejlYeEdU740GKQM7SaVS
# H3TbBL8R6HwX9QVpGnXPlKdE4fBIn5BBFnV+KwPxRNUNK6lYk2y1WSKour4hJN0S
# MkoaNV8hyyADiX1xuTxKaXN12HgR+8WulU2d6zhzXomJ2PleI9V2yfmfXSPGYanG
# gxzqI+ShoOGLomMd3mJt92nm7Mheng/TBeSA2z4I78JpwGpTRHiT7yHqBiV2ngUI
# yCtd0pZ8zg3S7bk4QC4RrcnKJ3FbjyPAGogmoiZ33c1HG93Vp6lJ415ERcC7bFQM
# RbxqrMVANiav1k425zYyFMyLNyE1QulQSgDpW9rtvVcIH7WvG9sqYup9j8z9J1Xq
# bBZPJ5XLln8mS8wWmdDLnBHXgYly/p1DhoQo5fkCAwEAAaOCAYswggGHMA4GA1Ud
# DwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMI
# MCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAfBgNVHSMEGDAWgBS6
# FtltTYUvcyl2mi91jGogj57IbzAdBgNVHQ4EFgQUn1csA3cOKBWQZqVjXu5Pkh92
# oFswWgYDVR0fBFMwUTBPoE2gS4ZJaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0Rp
# Z2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNybDCB
# kAYIKwYBBQUHAQEEgYMwgYAwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2lj
# ZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29t
# L0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNy
# dDANBgkqhkiG9w0BAQsFAAOCAgEAPa0eH3aZW+M4hBJH2UOR9hHbm04IHdEoT8/T
# 3HuBSyZeq3jSi5GXeWP7xCKhVireKCnCs+8GZl2uVYFvQe+pPTScVJeCZSsMo1JC
# oZN2mMew/L4tpqVNbSpWO9QGFwfMEy60HofN6V51sMLMXNTLfhVqs+e8haupWiAr
# SozyAmGH/6oMQAh078qRh6wvJNU6gnh5OruCP1QUAvVSu4kqVOcJVozZR5RRb/zP
# d++PGE3qF1P3xWvYViUJLsxtvge/mzA75oBfFZSbdakHJe2BVDGIGVNVjOp8sNt7
# 0+kEoMF+T6tptMUNlehSR7vM+C13v9+9ZOUKzfRUAYSyyEmYtsnpltD/GWX8eM70
# ls1V6QG/ZOB6b6Yum1HvIiulqJ1Elesj5TMHq8CWT/xrW7twipXTJ5/i5pkU5E16
# RSBAdOp12aw8IQhhA/vEbFkEiF2abhuFixUDobZaA0VhqAsMHOmaT3XThZDNi5U2
# zHKhUs5uHHdG6BoQau75KiNbh0c+hatSF+02kULkftARjsyEpHKsF7u5zKRbt5oK
# 5YGwFvgc4pEVUNytmB3BpIiowOIIuDgP5M9WArHYSAR16gc0dP2XdkMEP5eBsX7b
# f/MGN4K3HP50v/01ZHo/Z5lGLvNwQ7XHBx1yomzLP8lx4Q1zZKDyHcp4VQJLu2kW
# TsKsOqQwggauMIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEB
# CwUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNV
# BAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQg
# Um9vdCBHNDAeFw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNV
# BAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNl
# cnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIi
# MA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3
# qZdRodbSg9GeTKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOW
# bfhXqAJ9/UO0hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt
# 69OxtXXnHwZljZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3
# YYMZ3V+0VAshaG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECn
# wHLFuk4fsbVYTXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6Aa
# RyBD40NjgHt1biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTy
# UpURK1h0QCirc0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8U
# NM/STKvvmz3+DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCON
# WPfcYd6T/jnA+bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBAS
# A31fI7tk42PgpuE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp61
# 03a50g5rmQzSM7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAd
# BgNVHQ4EFgQUuhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJx
# XWRM3y5nP+e6mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUF
# BwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGln
# aWNlcnQuY29tMEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJo
# dHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNy
# bDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQEL
# BQADggIBAH1ZjsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CK
# Daopafxpwc8dB+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbP
# FXONASIlzpVpP0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaH
# bJK9nXzQcAp876i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxur
# JB4mwbfeKuv2nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/N
# h4cku0+jSbl3ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNB
# zU+2QJshIUDQtxMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77Qpf
# MzmHQXh6OOmc4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1Oby
# F5lZynDwN7+YAN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B
# 2RP+v6TR81fZvAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqk
# hQ/8mJb2VVQrH4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIFjTCCBHWg
# AwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0BAQwFADBlMQswCQYDVQQG
# EwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNl
# cnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVkIElEIFJvb3QgQ0EwHhcN
# MjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEVMBMG
# A1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEw
# HwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZwuEp
# pz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4VpX6+
# n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAdYykt
# zuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3T6cw
# 2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjUN6Qu
# BX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNdaSaTC
# 5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtmmnTK
# 3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyVw4/3
# IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3AeEP
# lAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYiCd98
# THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmpsh3l
# GwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU7NfjgtJx
# XWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNtyA8w
# DgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAkBggrBgEFBQcwAYYYaHR0
# cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0
# cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3J0MEUGA1Ud
# HwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFz
# c3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRVHSAAMA0GCSqGSIb3DQEB
# DAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyhhyzshV6pGrsi+IcaaVQi
# 7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre+i1Wz/n096wwepqL
# sl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo8L8vC6bp8jQ87PcDx4eo
# 0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++hUD38dglohJ9vytsgjTVg
# HAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5xaiNrIv8SuFQtJ37YOtnw
# toeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMYIDdjCCA3ICAQEwdzBjMQswCQYDVQQG
# EwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0
# IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBAhALrma8
# Wrp/lYfG+ekE4zMEMA0GCWCGSAFlAwQCAQUAoIHRMBoGCSqGSIb3DQEJAzENBgsq
# hkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjQxMDMxMDk0NTA4WjArBgsqhkiG
# 9w0BCRACDDEcMBowGDAWBBTb04XuYtvSPnvk9nFIUIck1YZbRTAvBgkqhkiG9w0B
# CQQxIgQgP+RpOqEF9XdYEcBx9hghz8vQPpruHDgFVu66hMXitF0wNwYLKoZIhvcN
# AQkQAi8xKDAmMCQwIgQgdnafqPJjLx9DCzojMK7WVnX+13PbBdZluQWTmEOPmtsw
# DQYJKoZIhvcNAQEBBQAEggIAsshgPmsgpFUlqm9Z+Hu+SGznkaKWKS2WOo4dn15e
# GfOySURS2+zRMTPjaQCsxX7VjuxSFQHDbpPHZ+36Xl/dKVnu6gWlNWcqT5cXPCGg
# 3pu9knT6NEOzIGJfagLBuHUd1Mljpy2MaVn17XnJvReMQ8u0GdKF5UhdKKj6U1uY
# WOLtEzXHlHf0QXnNwPaq4SbVHqS+6CwqEhy4rX8dtFygy8TKKSGr6fYE2dKWqp5n
# i2I6mSAZ4LfNnGcwUYozCu3Cmq22zxBhXWR33p3SKIPaG1/mCjawwwWBsx19L+Cv
# q6kIoYtr0VOMamDhWksDAbi/3hRzm3m7BGrIdMRZpJ6VjJzMnSLwfcIVGMY1GzoY
# 2j0VwPjrnlkhimtWwa+hB3h9e7hFmr1lkhsnUss156ViJsGC3/56/lTZmclaDKAT
# B2LPGcnDDe8tvbYQadlE+8+GhCtvnKbg3M1mIUxzCSslc34Evcl2LiN+nes7Ftu/
# X4w1v28xpzPq6LPsIyaQqhQxo63QshC5/zxoCqGEMhks7xrEF3DDaNoELxhgdKXx
# Vr4I/qPK0b9vvytHJN1yXH21lqV2TE3/Nuv9Dvi999Bh33NXmOldvckxiKhHnjKL
# M388Cif+vOQHxflgyf6o/K8EuIRh0Bs1DzmXPUOgcu1r7QhhjmU4uMsHLxIZTWvR
# q1E=
# SIG # End signature block
